import 'package:dio/dio.dart';
import 'package:flutter_dotenv/flutter_dotenv.dart';
import 'package:geocoding/geocoding.dart';
import 'package:geolocator/geolocator.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';
import 'package:razinsoft_rider/data/services/api/dio_client.dart';
import 'package:razinsoft_rider/data/services/local_storage_service.dart';

import '../../domain/interfaces/google_api_service_interface.dart';
import '../models/waypoint.dart';

class GoogleApiService implements IGoogleApiService {
  final DioClient dioClient;
  GoogleApiService({required this.dioClient});
  final String apiKey = dotenv.env['GOOGLE_MAPS_API_KEY']!;

  @override
  Future<Response> fetchWayPoints({required List<Waypoint> waypoints}) async {
    final String languageCode = await LocalStorageService()
        .getSelectedLanguage();

    final LatLng origin = waypoints.first.location;
    final LatLng destination = waypoints.last.location;
    final List<LatLng> stopPoints = waypoints
        .sublist(1, waypoints.length - 1)
        .map((e) => e.location)
        .toList();
    final String wayPointsString = stopPoints.isNotEmpty
        ? "&waypoints=${stopPoints.map((e) => "${e.latitude},${e.longitude}").join('|')}"
        : '';
    final String url =
        'https://maps.googleapis.com/maps/api/directions/json?origin=${origin.latitude},${origin.longitude}&destination=${destination.latitude},${destination.longitude}&language=$languageCode$wayPointsString&key=$apiKey';
    return dioClient.dio.get(url);
  }

  @override
  Future<Response> getAddressFromLatLng(LatLng latLng) async {
    final String languageCode = await LocalStorageService()
        .getSelectedLanguage();

    final String url =
        'https://maps.googleapis.com/maps/api/geocode/json?latlng=${latLng.latitude},${latLng.longitude}&language=$languageCode&key=$apiKey';
    return dioClient.dio.get(url);
  }

  @override
  Future<Response> getLatLngFromPlaceId(String placeId) async {
    final String languageCode = await LocalStorageService()
        .getSelectedLanguage();

    final String url =
        'https://maps.googleapis.com/maps/api/geocode/json?place_id=$placeId&language=$languageCode&key=$apiKey';
    return dioClient.dio.get(url);
  }

  @override
  Future<Response> searchPlace(String place) async {
    final String? countryCode = await getCurrentCountryCode();
    final String languageCode = await LocalStorageService()
        .getSelectedLanguage();

    final url =
        'https://maps.googleapis.com/maps/api/place/autocomplete/json'
        '?input=$place'
        '&language=$languageCode'
        '&components=country:$countryCode'
        '&key=$apiKey'; //&types=geocode
    return dioClient.dio.get(url);
  }

  @override
  Future<Response> getPlaceDetails(String placeId) async {
    final String languageCode = await LocalStorageService()
        .getSelectedLanguage();

    final String url =
        'https://maps.googleapis.com/maps/api/place/details/json?place_id=$placeId&language=$languageCode&key=$apiKey';
    return dioClient.dio.get(url);
  }

  @override
  Future<Response> fetchDistances(List<String> placeIds, LatLng origin) async {
    final String languageCode = await LocalStorageService()
        .getSelectedLanguage();

    final String destinations = placeIds.map((id) => 'place_id:$id').join('|');
    final String url =
        'https://maps.googleapis.com/maps/api/distancematrix/json'
        '?origins=${origin.latitude},${origin.longitude}'
        '&destinations=$destinations'
        '&language=$languageCode'
        // '&components=country:$countryCode'
        '&key=$apiKey';
    return dioClient.dio.get(url);
  }

  /// Step 1: Get current country code
  Future<String?> getCurrentCountryCode() async {
    LocationPermission permission = await Geolocator.checkPermission();
    if (permission == LocationPermission.denied) {
      permission = await Geolocator.requestPermission();
      if (permission == LocationPermission.denied) {
        return null;
      }
    }

    if (permission == LocationPermission.deniedForever) {
      return null;
    }

    final Position position = await Geolocator.getCurrentPosition(
      // desiredAccuracy: LocationAccuracy.high,
    );

    final List<Placemark> placemarks =
    await placemarkFromCoordinates(position.latitude, position.longitude);

    if (placemarks.isNotEmpty) {
      return placemarks.first.isoCountryCode?.toLowerCase();
    }

    return null;
  }

}
